{$O+,I-}
UNIT MSZ_ZMOD;

(* 

    MSZ_ZMOD - Zmodem code

    RENEWAVE is Copyright (C) 1994-2004 by Lars Hellsten and MatrixSoft(tm).

    This file is part of RENEWAVE.

    RENEWAVE is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RENEWAVE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RENEWAVE; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*)

INTERFACE


CONST MakeCrc32   : Boolean = TRUE;
      AllowResume : Boolean = TRUE;


PROCEDURE Zmodem_InitBuffers;
PROCEDURE Zmodem_DeinitBuffers;
FUNCTION  Zmodem_Receive(Path:String):Boolean;
FUNCTION  Zmodem_Send(FileSpec:String):Boolean;
PROCEDURE Zmodem_EndSend;


IMPLEMENTATION


USES  {$I FOSSTYPE.INC }

      {$IFDEF OS2}
      OS2BASE,
      {$ENDIF}

      {$IFDEF DELPHI}
      SysUtils,
      Windows,
      {$ENDIF}

      {$IFDEF DOS}
      MYSHARE,
      {$ENDIF}

      DOS,      CRT,      MSTRINGS, UNIXDATE, MISC1,
      CRC32,    FASTW,    MSZ_VID,  MSZ_MISC, TIMER;


CONST LastSent    : Byte = 0;
      TxTimeOut   : Byte = 10*18;
      RxTimeOut   : Byte = 100;

VAR   RxHdr       : tHeader;
      RxPos       : LongInt;
      RxType      : Integer;
      RxFrameInd  : Integer;
      RxBufLen    : Integer;
      RxHdrType   : Byte;     { Header type }
      RxConvType  : Byte;     { Conversion options }
      RxCount     : Integer;  { Position in buffer }
      RxPath      : String;

VAR   TxBuf       : pBuffer;
      TxPos       : LongInt;
      TxHdr       : tHeader;

      SendCrc32   : Boolean;

VAR   Attn        : String;

VAR   FileName    : String;
      FileTime    : LongInt;


{$I ZCONST.INC }


PROCEDURE Zmodem_InitBuffers;
BEGIN
   GetMem(TxBuf,8192);
   GetMem(DataBuf,8192);
END;


PROCEDURE Zmodem_DeInitBuffers;
BEGIN
   FreeMem(DataBuf,8192);
   FreeMem(TxBuf,8192);
END;


FUNCTION Zmodem_TimedRead:Integer;
{ Ignores XON/XOFF characters }
VAR Stop:Boolean; Ch:Char; Time:LongInt;
BEGIN
   Time := TimeCounter+(RxTimeOut*100);
   Stop := FALSE;
   REPEAT
      IF KeyPressed THEN IF (ReadKey = #27) THEN
         BEGIN
            MSZ_SendCan;
            Zmodem_Message('Aborted from keyboard');
            Zmodem_TimedRead := ABORT;
            Exit;
         END;
      IF fk_RemoteKeyPressed THEN
         BEGIN
            ch := Ms_ReadChar;
            IF (ch <> Chr(XON)) AND (ch <> Chr(XOFF)) THEN Stop := TRUE; { Strip XON, etc. }
         END;
   UNTIL Stop OR (TimeCounter > Time) OR (Ms_CheckCarrier=FALSE);
   IF (TimeCounter > Time) AND (Stop=FALSE)
      THEN Zmodem_TimedRead := TIMEDOUT
      ELSE IF (Ms_CheckCarrier=FALSE) AND (Stop=FALSE)
         THEN Zmodem_TimedRead := RCDO
         ELSE Zmodem_TimedRead := Ord(Ch);
END;


PROCEDURE Zmodem_PutHex(b:Byte);
{ Output a byte as two hex digits (in ASCII), uses lower case to avoid }
{ confusion with escaped control characters.                           }
CONST HexDigits:Array[0..15] OF Char = '0123456789abcdef';
BEGIN
   Ms_WriteChar(HexDigits[b SHR 4]);
   Ms_WriteChar(HexDigits[b AND $0F]);
END;


PROCEDURE Zmodem_SendHexHeader(hType:Byte; VAR Header:tHeader);
{ Sends a zmodem hex type header }
VAR Crc:Word; i,n:Integer;
BEGIN
   Ms_WriteString(Chr(ZPAD)+Chr(ZPAD)+Chr(ZDLE)+Chr(ZHEX));
   Zmodem_PutHex(hType);
   Crc := UpdC16(hType,0);
   FOR n := 0 TO 3 DO
      BEGIN
         Zmodem_PutHex(Header[n]);
         Crc := UpdC16(Header[n],Crc)
      END;
   Crc := UpdC16(0,Crc);
   Crc := UpdC16(0,Crc);
   Zmodem_PutHex(Lo(Crc SHR 8));
   Zmodem_PutHex(Lo(Crc));
   Ms_WriteString(#13#10);
   IF (hType <> ZFIN) AND (hType <> ZACK) THEN Ms_WriteChar(#17);
END;


FUNCTION Zmodem_PullLongFromHeader(VAR Header:tHeader):LongInt;
{ Stuffs a longint into a header variable - bytes are REVERSED! }
TYPE LongArray = Array[0..3] OF BYTE;
VAR  l         : LongInt;
     LongPtr   : LongArray ABSOLUTE l;
BEGIN
   LongPtr[0] := Header[ZP0];
   LongPtr[1] := Header[ZP1];
   LongPtr[2] := Header[ZP2];
   LongPtr[3] := Header[ZP3];

   Zmodem_PullLongFromHeader := l
END;


PROCEDURE Zmodem_PutLongIntoHeader(l:LongInt);
TYPE LongArray = Array[0..3] OF BYTE;
VAR  LongPtr   : LongArray ABSOLUTE l;
BEGIN
   txHdr[ZP0] := LongPtr[0];
   txHdr[ZP1] := LongPtr[1];
   txHdr[ZP2] := LongPtr[2];
   txHdr[ZP3] := LongPtr[3];
END;


FUNCTION Zmodem_GetZDL:Integer;
{ Gets a byte and processes for ZDLE escaping or CAN sequence }
VAR c:Integer;
BEGIN
   c := MSZ_TimedRead(rxTimeout);
   IF (c <> ZDLE)
      THEN Zmodem_GetZDL := c
      ELSE BEGIN
            c := MSZ_TimedRead(rxTimeout); IF (c = CAN) THEN BEGIN { got 2nd CAN }
            c := MSZ_TimedRead(rxTimeout); IF (c = CAN) THEN BEGIN { got 3rd CAN }
            c := MSZ_TimedRead(rxTimeout); IF (c = CAN) THEN BEGIN
            c := MSZ_TimedRead(rxTimeout); END; END; END;

            CASE c OF
               CAN    : Zmodem_GetZDL := GOTCAN; { Got 5th CAN }
               ZCRCE,
               ZCRCG,
               ZCRCQ,
               ZCRCW  : Zmodem_GetZDL := (c OR GOTOR);
               ZRUB0  : Zmodem_GetZDL := $007F; { DEL }
               ZRUB1  : Zmodem_GetZDL := $00FF;
               ELSE     BEGIN
                           IF (c < 0)
                              THEN Zmodem_GetZDL := c
                              ELSE IF ((c AND $60) = $40)
                                 THEN Zmodem_GetZDL := c XOR $40
                                 ELSE Zmodem_GetZDL := ZERROR;
                        END;
            END;
         END;
END;


FUNCTION Zmodem_GetHex:Integer;
(* Get a byte that has been received as two ASCII hex digits *)
VAR i,j:Integer;
BEGIN
   i := Zmodem_TimedRead;
   IF (i < 0) THEN
      BEGIN
         Zmodem_GetHex := i;
         Exit;
      END;
   i := i-$30;
   IF (i > 9) THEN i := i-39;
   IF (i AND $FFF0 <> 0) THEN { If not in range 0-F (hex digit) }
      BEGIN
         Zmodem_GetHex := ZERROR;
         Exit;
      END;
   j := Zmodem_TimedRead;
   IF (j < 0) THEN
      BEGIN
         Zmodem_GetHex := j;
         Exit;
      END;
   j := j - $30;
   IF (j > 9) THEN j := j - 39;
   IF (j AND $FFF0 <> 0) THEN
      BEGIN
         Zmodem_GetHex := ZERROR;
         Exit;
      END;
   Zmodem_GetHex := (i SHL 4) OR j;
END;


FUNCTION Zmodem_GetHexHeader(VAR hdr:tHeader):Integer;
VAR Crc:Word; i,j:Integer;
BEGIN
   i := Zmodem_GetHex;
   IF (i < 0) THEN
      BEGIN
         Zmodem_GetHexHeader := i;
         Exit;
      END;
   rxType := i;
   Crc := UpdC16(Lo(rxType),0);
   FOR j := 0 To 3 DO                  {get the 4 bytes}
      BEGIN
         i := Zmodem_GetHex;
         IF (i < 0) THEN
            BEGIN
               Zmodem_GetHexHeader := i;
               Exit;
            END;
         hdr[j] := Lo(i);
         Crc := UpdC16(Lo(i),Crc)
      END;

   i := Zmodem_GetHex;
   IF (i < 0) THEN
      BEGIN
         Zmodem_GetHexHeader := i;
         Exit;
      END;
   Crc := UpdC16(Lo(i),Crc);

   i := Zmodem_GetHex;
   IF (i < 0) THEN
      BEGIN
         Zmodem_GetHexHeader := i;
         Exit;
      END;
   crc := UpdC16(Lo(i),crc);             {check the CRC}

   IF (crc <> 0) THEN
      BEGIN
         Zmodem_Message(StrFunc(RxPos)+': Bad CRC in ZHEX header!');
         Inc(Errors);
         Zmodem_Errors(Errors);
         Zmodem_GetHexHeader := ZERROR;
         Exit;
      END;

   IF (MSZ_TimedRead(2) = 13) THEN i := MSZ_TimedRead(2);
   Zmodem_GetHexHeader := rxType;
END;


FUNCTION Zmodem_GetBinaryHeader(VAR hdr:tHeader):Integer;
VAR Crc:Word; i,j:Integer;
BEGIN
   i := Zmodem_GetZDL;
   IF (i < 0) THEN
      BEGIN
         Zmodem_GetBinaryHeader := i;
         Exit;
      END;

   rxtype := i;
   crc := UpdC16(Lo(rxtype),0);

   FOR j := 0 TO 3 DO
      BEGIN
         i := Zmodem_GetZDL;
         IF (Hi(i) <> 0) THEN
            BEGIN
               Zmodem_GetBinaryHeader := i;
               Exit;
            END;
         hdr[j] := Lo(i);
         crc := UpdC16(Lo(i),crc)
      END;

   i := Zmodem_GetZDL;
   IF (Hi(i) <> 0) THEN
      BEGIN
         Zmodem_GetBinaryHeader := i;
         Exit
      END;
   crc := UpdC16(Lo(i),crc);

   i := Zmodem_GetZDL;
   IF (Hi(i) <> 0) THEN
      BEGIN
         Zmodem_GetBinaryHeader := i;
         Exit;
      END;
   crc := UpdC16(Lo(i),crc);

   IF (crc <> 0) THEN
      BEGIN
         Zmodem_Message(StrFunc(RxPos)+': Bad CRC in ZBIN16 header!');
         Inc(Errors);
         Zmodem_Errors(Errors);
         Exit
      END;
   Zmodem_GetBinaryHeader := rxType
END;


FUNCTION Zmodem_GetBinaryHead32(VAR hdr:tHeader):Integer;
VAR Crc:LongInt; i,j:Integer;
BEGIN
   i := Zmodem_GetZDL;
   IF (i < 0) THEN
      BEGIN
         Zmodem_GetBinaryHead32 := i;
         Exit;
      END;

   rxType := i;
   Crc := UpdC32(Lo(rxType),$FFFFFFFF);

   FOR j := 0 TO 3 DO
      BEGIN
         i := Zmodem_GetZDL;
         IF (Hi(i) <> 0) THEN
            BEGIN
               Zmodem_GetBinaryHead32 := i;
               Exit;
            END;
         hdr[j] := Lo(i);
         crc := UpdC32(Lo(i),crc)
      END;

   FOR j := 0 TO 3 DO
      BEGIN
         i := Zmodem_GetZDL;
         IF (Hi(i) <> 0) THEN
            BEGIN
               Zmodem_GetBinaryHead32 := i;
               Exit;
            END;
         crc := UpdC32(Lo(i),crc)
      END;

   IF (crc <> $DEBB20E3) THEN { The polynomial value }
      BEGIN
         Inc(Errors);
         Zmodem_Message(StrFunc(RxPos)+': Bad CRC in ZBIN32 header!');
         Zmodem_Errors(Errors);
         Zmodem_GetBinaryHead32 := ZERROR;
         Exit;
      END;

   Zmodem_GetBinaryHead32 := rxType
END;


FUNCTION Zmodem_GetHeader(VAR hdr:tHeader):Integer;
{ Gets a header - will figure out what type it is (hex,bin16,bin32) and }
{ call the appropriate routine }
LABEL gotcancel, again, again2, splat, done;
VAR   c, n, cancount:Integer;
BEGIN
   IF fk_Fossil.Baud > $3FFF
      THEN n := $7FFF
      ELSE n := fk_Fossil.Baud*2; { Guess the # of garbage chars expected }

   CanCount := 5;
   SendCrc32 := FALSE; { Assume 16 bit until proven otherwise }

AGAIN:

   rxFrameind := 0;
   rxType := 0;

   c := Zmodem_TimedRead;

   CASE c OF
      ZPAD     : { All headers begin with '*' } ;
      ABORT,
      RCDO,
      TIMEDOUT : GOTO Done;
      CAN      : BEGIN

GOTCANCEL:

                    Dec(cancount);
                    IF (cancount < 0) THEN
                       BEGIN
                          c := ZCAN;
                          GOTO Done;
                       END;
                    c := MSZ_TimedRead(2);
                    CASE c OF
                       TIMEDOUT : GOTO Again;
                       ZCRCW    : BEGIN
                                     c := ZERROR;
                                     GOTO done;
                                  END;
                       ABORT,
                       RCDO     : GOTO Done;
                       CAN      : BEGIN
                                     Dec(cancount);
                                     IF (cancount < 0) THEN
                                        BEGIN
                                           c := ZCAN;
                                           GOTO done
                                        END;
                                     GOTO Again;
                                  END;
                    END;
                 END
      ELSE
AGAIN2:          BEGIN
                    Dec(n);
                    IF (n < 0) THEN
                       BEGIN
                          Inc(Errors);
                          Zmodem_Errors(Errors);
                          Zmodem_Message(StrFunc(RxPos)+': Garbled header');
                          Zmodem_GetHeader := ZERROR;
                          Exit;
                       END;
                    IF (c <> CAN) THEN cancount := 5;
                    GOTO Again;
                 END;
   END;
   { The above only falls through if ZPAD - anything else is shit }

   cancount := 5;

SPLAT:

   c := Zmodem_TimedRead;
   CASE c OF
      ZDLE     : { We want this next ... } ;
      ZPAD     : GOTO Splat; { Junk or second '*' of a hex header }
      ABORT,
      RCDO,
      TIMEDOUT : GOTO Done;
      ELSE       GOTO Again2;
   END;
   { Only falls through if c is ZDLE }

   c := Zmodem_TimedRead;
   CASE c OF
      ZBIN32   : BEGIN
                    rxFrameInd := ZBIN32; { Use 32 bit CRC }
                    c := Zmodem_GetBinaryHead32(hdr);
                 END;
      ZBIN     : BEGIN
                    rxFrameInd := ZBIN; { Bin with 16 bit CRC }
                    c := Zmodem_GetBinaryHeader(hdr)
                 END;
      ZHEX     : BEGIN
                    rxFrameInd := ZHEX; { Hex }
                    c := Zmodem_GetHexHeader(hdr);
                 END;
      CAN      : GOTO GotCancel;
      ABORT,
      RCDO,
      TIMEDOUT : GOTO Done;
      ELSE       GOTO Again2;
   END;
   { Only falls through if we get ZBIN32, ZBIN or ZHEX }

   rxPos := Zmodem_PullLongFromHeader(hdr);

DONE:

   Zmodem_GetHeader := c;
END;


FUNCTION RZ_ReceiveData(Buf:pBuffer; bLength:Integer):Integer;
LABEL crcfoo;
VAR   c, d      : Integer;
      n         : Word;
      Crc16     : Word;
      Crc32     : LongInt;
      UsesCrc32 : Boolean;
      BadCrc    : Boolean;
      Done      : Boolean;
BEGIN
   IF (rxFrameInd = ZBIN32)
      THEN BEGIN
            Crc32 := $FFFFFFFF;
            UsesCrc32 := TRUE;
            Zmodem_ShowCheck('Zmodem CRC-32/'+StrFunc(ZBUFSIZE));
         END
      ELSE BEGIN
            Crc16 := 0;
            UsesCrc32 := FALSE;
            Zmodem_ShowCheck('Zmodem CRC-16/'+StrFunc(ZBUFSIZE));
         END;

   RxCount := 0;
   Done := FALSE;

   REPEAT
      c := Zmodem_GetZDL;
      IF (Hi(c) <> 0)
         THEN BEGIN
               IF KeyPressed THEN IF ReadKey=#27 THEN
                  BEGIN
                     MSZ_SendCan;
                     Zmodem_Message('Aborted from keyboard!');
                     RZ_ReceiveData := ZCAN;
                     Exit;
                  END;
               Done := TRUE;
CRCFOO:        CASE c OF
                  GOTCRCE,
                  GOTCRCG,
                  GOTCRCQ,
                  GOTCRCW   : BEGIN { CRC's }
                                 d := c;
                                 IF UsesCrc32
                                    THEN BEGIN
                                          Crc32 := UpdC32(Lo(c),Crc32);
                                          FOR n := 0 TO 3 DO
                                             BEGIN
                                                c := Zmodem_GetZDL;
                                                IF (c < 0) THEN GOTO CrcFoo;
                                                Crc32 := UpdC32(Lo(c),Crc32)
                                             END;
                                          BadCrc := (Crc32 <> $DEBB20E3);
                                       END
                                    ELSE BEGIN
                                          Crc16 := UpdC16(Lo(c),Crc16);
                                          c := Zmodem_GetZDL;
                                          IF (c < 0) THEN GOTO CrcFoo;
                                          Crc16 := UpdC16(Lo(c),Crc16);
                                          c := Zmodem_GetZDL;
                                          IF (c < 0) THEN GOTO CrcFoo;
                                          Crc16 := UpdC16(Lo(c),Crc16);
                                          BadCrc := (Crc16 <> 0);
                                       END;
                                 IF BadCrc
                                    THEN BEGIN
                                          IF UsesCrc32
                                             THEN Zmodem_Message(StrFunc(RxPos)+': Bad CRC ('+LongInt2Hex(Crc32)+'h)')
                                             ELSE Zmodem_Message(StrFunc(RxPos)+': Bad CRC ('+Int2Hex(Crc16)+'h)');
                                          RZ_ReceiveData := ZERROR;
                                       END
                                    ELSE RZ_ReceiveData := d;
                              END;
                  GOTCAN    : BEGIN
                                 Zmodem_Message('Transfer aborted by remote!');
                                 RZ_ReceiveData := ZCAN;
                              END;
                  TIMEDOUT  : BEGIN
                                 RZ_ReceiveData := c;
                                 Zmodem_Message(StrFunc(RxPos)+': Timed out!');
                              END;
                  RCDO      : BEGIN
                                 Zmodem_Message('Lost carrier!');
                                 RZ_ReceiveData := c;
                              END;
                  ABORT     : RZ_ReceiveData := c;
                  ELSE        BEGIN
                                 Done := FALSE;
                                 RZ_ReceiveData := c;
                              END;
               END;
            END
         ELSE BEGIN
               Dec(bLength);
               IF (bLength < 0)
                  THEN BEGIN
                        Zmodem_Message(StrFunc(RxPos)+': Long (garbled) packet!');
                        RZ_ReceiveData := ZERROR;
                        Done := TRUE;
                     END
                  ELSE BEGIN
                        Buf^[Integer(RxCount)] := Lo(c);
                        Inc(RxCount);
                        IF UsesCrc32
                           THEN Crc32 := UpdC32(Lo(c),Crc32)
                           ELSE Crc16 := UpdC16(Lo(c),Crc16);
                     END;
            END;
   UNTIL Done;
END;


PROCEDURE RZ_AckBibi;
{ Ack a ZFIN packet - other end's request to terminate cleanly }
VAR n:Integer;
BEGIN
   Zmodem_PutLongIntoHeader(RxPos);
   n := 4;
   fk_PurgeInputBuffer;
   REPEAT
      Zmodem_SendHexHeader(ZFIN,txHdr);
      CASE MSZ_TimedRead(2) OF
         TIMEDOUT  : n := 0;
         RCDO      : n := 0;
         ABORT     : n := 0;
         79 {'O'}  : BEGIN { Discard second 'O' }
                        fk_PurgeInputBuffer;
                        n := 0;
                     END;
      END;
      Dec(n);
   UNTIL (n <= 0);
END; { RZ_AckBibi }


FUNCTION RZ_InitReceiver:Integer;
VAR   c,
      n,
      Errors : Integer;
      Stop   : Boolean;
      Again  : Boolean;
BEGIN
   Attn := '';
   n := 20;
   Stop := FALSE;
   Errors := 0;

   WHILE (n > 0) AND (Stop=FALSE) DO
      BEGIN
         IF (Ms_CheckCarrier=FALSE) THEN
            BEGIN
               Zmodem_Message('Lost carrier!');
               RZ_InitReceiver := ZERROR;
               Exit;
            END;
         Zmodem_PutLongIntoHeader(0);
         TxHdr[ZF0] := CANFDX OR CANOVIO OR CANBRK; { Full duplex, overlay I/O }
         IF MakeCrc32 THEN TxHdr[ZF0] := TxHdr[ZF0] OR CANFC32;
         Zmodem_SendHexHeader(RxHdrType,TxHdr);
         IF (RxHdrType = ZSKIP) THEN RxHdrType := ZRINIT;
         Again := FALSE;
         REPEAT
            c := Zmodem_GetHeader(rxhdr);
            Zmodem_Frame(c);

            CASE c OF
               ZFILE    : BEGIN
                             RxConvType := RxHdr[ZF0];
                             RxHdrType := ZRINIT;
                             c := RZ_ReceiveData(DataBuf,ZBUFSIZE);
                             Zmodem_Frame(c);
                             IF (c = GOTCRCW)
                                THEN BEGIN
                                      RZ_InitReceiver := ZFILE;
                                      Stop := TRUE;
                                   END
                                ELSE BEGIN
                                      Zmodem_SendHexHeader(ZNAK,txhdr);
                                      Again := TRUE;
                                   END;
                          END;
               ZSINIT   : BEGIN
                             c := RZ_ReceiveData(DataBuf,ZBUFSIZE);
                             Zmodem_Frame(c);
                             FillChar(DataBuf^,ZBUFSIZE,0);
                             IF (c = GOTCRCW)
                                THEN Zmodem_SendHexHeader(ZACK,TxHdr)
                                ELSE Zmodem_SendHexHeader(ZNAK,TxHdr);
                             Again := TRUE;
                          END;
               ZFREECNT : BEGIN
                             Zmodem_PutLongIntoHeader(DiskFree(0));
                             Zmodem_SendHexHeader(ZACK,txhdr);
                             Again := TRUE;
                          END;
               ZCOMMAND : BEGIN
                             c := RZ_ReceiveData(DataBuf,ZBUFSIZE);
                             Zmodem_Frame(c);
                             IF (c = GOTCRCW)
                                THEN BEGIN
                                      Zmodem_PutLongIntoHeader(0);
                                      Errors := 0;
                                      REPEAT
                                         Zmodem_SendHexHeader(ZCOMPL,TxHdr);
                                         Inc(Errors)
                                      UNTIL (Errors > 10) OR (Zmodem_GetHeader(rxHdr) = ZFIN);
                                      RZ_AckBibi;
                                      RZ_InitReceiver := ZCOMPL;
                                      Stop := TRUE;
                                   END
                                ELSE BEGIN
                                      Zmodem_SendHexHeader(ZNAK,txhdr);
                                      Again := TRUE;
                                   END;
                          END;
               ZCOMPL,
               ZFIN     : BEGIN
                             RZ_InitReceiver := ZCOMPL;
                             Stop := TRUE
                          END;
               ABORT,
               ZCAN,
               RCDO     : BEGIN
                             RZ_InitReceiver := c;
                             Stop := TRUE;
                          END;
            END;
         UNTIL (Again=FALSE) OR Stop;
         IF NOT Stop THEN RealDelay(1000);
         Dec(n);
      END;
   IF (Stop=FALSE) THEN
      BEGIN
         Zmodem_Message(StrFunc(RxPos)+': Timed out!');
         RZ_InitReceiver := ZERROR;
      END;
END;


FUNCTION RZ_GetHeader:Integer;
VAR   ReturnCode  : Integer;
      e, p, n     : Integer;
      Multiplier  : LongInt;
      MakeFile    : Boolean;
      i           : Byte;
      s           : String;
      ttime,
      tsize       : LongInt;
      tname       : String;

BEGIN
   MakeFile := FALSE;
   StartTime := CurrentSecsFunc;
   ReturnCode := ZOK;
   MSZ_GetInfoHeader(DataBuf,tName,FileTotal,FileTime);
   IF FileName = '' THEN FileName := tName;
   IF MSZ_CheckBadName(FileName) <> '' THEN
      BEGIN
         Zmodem_Message('Tried to upload to '+MSZ_CheckBadName(FileName)+' - skipping!');
         RealDelay(1000);
         RZ_GetHeader := ZSKIP;
         Exit;
      END;

   IF AllowResume AND (Zmodem_FindFile(RxPath+FileName,tName,tSize,tTime))
      THEN BEGIN
            IF (tTime = FileTime)
               THEN BEGIN
                     IF (RxConvType = ZCRESUM) AND (FileTotal = tSize)
                        THEN BEGIN
                              Zmodem_ShowName(RxPath+FileName);
                              Zmodem_Message('File is already complete, skipping');
                              ReturnCode := ZSKIP;
                           END
                        ELSE IF (FileTotal > tSize) THEN
                           BEGIN
                              FileTxStart := tSize;
                              Assign(TheFile,RxPath+FileName);
                              Reset(TheFile,1);
                              IF (IOResult <> 0)
                                 THEN BEGIN
                                       Zmodem_Message('Error opening '+FileName);
                                       ReturnCode := ZERROR;
                                    END
                                 ELSE BEGIN
                                       Seek(TheFile,tSize);
                                       IF (IOResult <> 0)
                                          THEN BEGIN
                                                Zmodem_Message('Error positioning file');
                                                ReturnCode := ZERROR;
                                             END
                                          ELSE BEGIN
                                                Zmodem_Message('Resuming from offset '+StrFunc(tSize));
                                                FileAddition := RecoverFile;
                                             END;
                                    END;
                           END
                        ELSE BEGIN
                              MakeFile := TRUE;
                              FileAddition := ReplaceFile;
                           END;
                  END
               ELSE BEGIN
                     MakeFile := TRUE;
                     FileAddition := ReplaceFile;
                  END;
         END
      ELSE BEGIN
            MakeFile := TRUE;
            FileAddition := NewFile;
         END;

   IF MakeFile THEN
      BEGIN
         FileTxStart := 0;
         Assign(TheFile,RxPath+FileName);
         Rewrite(TheFile,1);
         IF IOResult <> 0 THEN
            BEGIN
               Zmodem_Message('Unable to create '+FileName);
               ReturnCode := ZERROR;
            END;
      END;

   Zmodem_ShowName(RxPath+FileName);
   Zmodem_ShowSize;
   RZ_GetHeader := ReturnCode;
END;


FUNCTION RZ_SaveToDisk(VAR rxBytes:LongInt):Integer;
VAR {$IFDEF OS2} BytesWritten:LongInt; {$ELSE} BytesWritten:Integer; {$ENDIF}
BEGIN
   Zmodem_ShowBlockSize(rxCount);
   BlockWrite(TheFile,DataBuf^,rxCount,BytesWritten);
   IF BytesWritten <> rxCount
      THEN BEGIN
            Zmodem_Message(StrFunc(RxPos)+': Disk write error');
            RZ_SaveToDisk := ZERROR;
         END
      ELSE RZ_SaveToDisk := 0;
   rxBytes := rxBytes + rxCount
END;


FUNCTION RZ_ReceiveFile:Integer;
LABEL Error, NextHeader, MoreData; { UGH - wouldn't be necessary in C }
VAR   c, n    : Integer;
      rxbytes : LongINt;
      sptr    : String;
      Done    : Boolean;
      NumStr  : String[10];

      FUNCTION SaveDataBlock:Integer;
      VAR i:LongInt;
      BEGIN
         n := 10;
         i := RxBytes;
         SaveDataBlock := RZ_SaveToDisk(RxBytes);
         FileTx := RxBytes;
         Zmodem_ShowLoc;
         Zmodem_ShowBatchLoc;
      END;

BEGIN
   Done := TRUE;

   c := RZ_GetHeader;
   IF (c <> ZOK) THEN
      BEGIN
         IF (c = ZSKIP) THEN RxHdrType := ZSKIP;
         RZ_ReceiveFile := c;
         Exit;
      END;

   n := 10;
   rxbytes := FileTxStart;
   rxpos := FileTxStart;

   Cps := 0;
   Zmodem_Message('Receiving file');

   REPEAT
      fk_PurgeOutputBuffer;
      Zmodem_PutLongIntoHeader(rxbytes);
      Zmodem_SendHexHeader(ZRPOS,txhdr);
      fk_PurgeInputBuffer;

NextHeader:

      c := Zmodem_GetHeader(rxhdr);
      Zmodem_Frame(c);

      CASE c OF
         ZDATA    : BEGIN { Data packet }
                       IF (rxpos <> rxbytes)
                          THEN BEGIN
                                Dec(n);
                                Inc(Errors);
                                Zmodem_Errors(Errors);
                                IF (n < 0) THEN GOTO Error;
                                Zmodem_Message(StrFunc(RxPos)+': Bad position');
                                Ms_WriteString(Attn);
                             END
                          ELSE BEGIN

MoreData:
                                FileTx := RxBytes;
                                Zmodem_ShowLoc;
                                Zmodem_ShowBatchLoc;
                                c := RZ_ReceiveData(DataBuf,ZBUFSIZE);
                                Zmodem_Frame(c);

                                CASE c OF
                                   ABORT    : GOTO Error;
                                   ZCAN     : GOTO Error;
                                   RCDO     : GOTO Error;
                                   ZERROR   : BEGIN
                                                 Dec(n);
                                                 Inc(Errors);
                                                 Zmodem_Errors(Errors);
                                                 IF (n < 0) THEN GOTO Error;
                                              END;
                                   TIMEDOUT : BEGIN
                                                 Dec(n);
                                                 Inc(Errors);
                                                 Zmodem_Errors(Errors);
                                                 Zmodem_Message(StrFunc(RxBytes)+': Timeout');
                                                 IF (n < 0) THEN GOTO Error;
                                              END;
                                   GOTCRCW  : BEGIN { End of frame }
                                                 c := SaveDataBlock;
                                                 IF (c <> 0) THEN Exit;
                                                 Zmodem_PutLongIntoHeader(rxbytes);
                                                 Zmodem_SendHexHeader(ZACK,txhdr);
                                                 GOTO NextHeader;
                                              END;
                                   GOTCRCQ  : BEGIN { ZACK expected }
                                                 c := SaveDataBlock;
                                                 IF (c <> 0) THEN Exit;
                                                 Zmodem_PutLongIntoHeader(rxbytes);
                                                 Zmodem_SendHexHeader(ZACK,txhdr);
                                                 GOTO MoreData;
                                              END;
                                   GOTCRCG  : BEGIN { Non-stop }
                                                 c := SaveDataBlock;
                                                 IF (c <> 0) THEN Exit;
                                                 GOTO MoreData;
                                              END;
                                   GOTCRCE  : BEGIN { Header to follow }
                                                 c := SaveDataBlock;
                                                 IF (c <> 0) THEN Exit;
                                                 GOTO NextHeader;
                                              END;
                                END;
                             END;
                    END;
         ZNAK,
         TIMEDOUT : BEGIN { Packet was probably garbled }
                       Dec(n);
                       IF (n < 0) THEN
                          BEGIN
                             c := ZERROR;
                             GOTO Error;
                          END;
                    END;
         ZFILE    : BEGIN { Sender didn't see our ZRPOS yet }
                       c := RZ_ReceiveData(DataBuf,ZBUFSIZE);
                       Zmodem_Frame(c)
                    END;
         ZEOF     : IF (rxpos = rxbytes)  { End of the file }
                       THEN BEGIN
                             RZ_ReceiveFile := c;
                             Exit;
                          END
                       ELSE GOTO NextHeader;
         ZERROR   : BEGIN { Too much garbage in header search error }
                       Dec(n);
                       IF (n < 0) THEN GOTO Error;
                       Ms_WriteString(Attn);
                    END;
         ABORT    : GOTO Error;
         ELSE       BEGIN
                       c := ZERROR;
                       GOTO Error;
                    END;
      END;
      FileTx := RxBytes;
      Zmodem_ShowLoc;
      Zmodem_ShowBatchLoc;
   UNTIL (Done=FALSE);

Error:

   FileTx := RxBytes;
   Zmodem_ShowLoc;
   Zmodem_ShowBatchLoc;
   RZ_ReceiveFile := c;
END;


FUNCTION RZ_ReceiveBatch:Integer;
VAR s,FilePath:String; c:Integer; Done:Boolean;
BEGIN
   Done := FALSE;
   WHILE (Done=FALSE) DO
      BEGIN
         IF (Ms_CheckCarrier=FALSE) THEN
            BEGIN
               RZ_ReceiveBatch := ZERROR;
               Exit;
            END;

         StartTime := CurrentSecsFunc;
         Zmodem_InitWindow(FALSE);
         Zmodem_Message('Sending ZRPOS; ready to receive file');
         c := RZ_ReceiveFile;
         Zmodem_Frame(c);
         SetFTime(TheFile,FileTime);
         Close(TheFile);
         FilePath := RxPath+FileName;
         FileName := '';
         BatchTx := BatchTx+FileTotal;
         Inc(BatchCurrent);
         FileTx := RxPos;

         CASE c OF
            ZEOF,
            ZSKIP : BEGIN
                       c := RZ_InitReceiver;
                       Zmodem_Frame(c);
                       CASE c OF
                          ZFILE  : BEGIN
                                      MSZ_WriteLog('Z',FileTx,FilePath,TRUE);
                                      Zmodem_Message('Waiting for file');
                                      FileAddition := NewFile;
                                   END;
                          ZCOMPL : BEGIN
                                      MSZ_WriteLog('Z',FileTx,FilePath,TRUE);
                                      Zmodem_Message('Transfer complete');
                                      RZ_AckBibi;
                                      RZ_ReceiveBatch := ZOK;
                                      Exit;
                                   END;
                          ELSE     BEGIN
                                      MSZ_WriteLog('Z',FileTx,FilePath,FALSE);
                                      RZ_ReceiveBatch := ZERROR;
                                      Exit;
                                   END;
                       END;
                    END;
            ELSE    BEGIN
                       MSZ_WriteLog('Z',FileTx,FilePath,c=ZOK);
                       RZ_ReceiveBatch := c;
                       Exit;
                    END;
         END;
{         Zmodem_ShowLoc;
         Zmodem_ShowBatchLoc;}
      END; { While }
END;


FUNCTION Zmodem_Receive(Path:String):Boolean;
VAR i:Integer;
BEGIN
   Zmodem_Message('Initializing...');
   FileName := '';
   IF DirExists(Path)
      THEN Path := AddSlash(Path)
      ELSE FileName := ExtractFName(Path);
   RxPath := ExtractFDir(Path);
   IF RxPath <> '' THEN RxPath := AddSlash(RxPath);
   rxTimeOut := 10;
   RxHdrType := ZRINIT;
   i := RZ_InitReceiver;
   IF (i = ZFILE) THEN i := RZ_ReceiveBatch;
   IF (i = ZCOMPL) OR (i = ZOK)
      THEN Zmodem_Receive := TRUE
      ELSE BEGIN
            fk_PurgeOutputBuffer;
            Zmodem_Message('Transfer unsuccessful, sending CAN');
            MSZ_SendCan;
            Zmodem_Receive := FALSE;
            RealDelay(2000);
         END;
END;


{ SEND ROUTINES }


PROCEDURE SZ_SendByte(b:Byte);
BEGIN
   IF ((b AND $7F) IN [16,17,19,24]) OR (((b AND $7F) = 13) AND ((LastSent AND $7F) = 64))
      THEN BEGIN
            Ms_WriteChar(Chr(ZDLE));
            LastSent := (b XOR $40)
         END
      ELSE LastSent := b;
   Ms_WriteChar(Chr(LastSent));
END;


PROCEDURE SZ_SendBinaryHeader(htype: BYTE; VAR hdr:tHeader);
VAR Crc16 : Word;
    Crc32 : LongInt;
    n     : Integer;
BEGIN
   Ms_WriteChar(Chr(ZPAD));
   Ms_WriteChar(Chr(ZDLE));

   IF SendCrc32
      THEN BEGIN
            Ms_WriteChar(Chr(ZBIN32));
            SZ_SendByte(htype);
            Crc32 := UpdC32(hType,$FFFFFFFF);
            FOR n := 0 TO 3 DO
               BEGIN
                  SZ_SendByte(hdr[n]);
                  Crc32 := UpdC32(hdr[n],Crc32)
               END;
            Crc32 := NOT Crc32;
            FOR n := 0 TO 3 DO
               BEGIN
                  SZ_SendByte(Byte(Crc32));
                  Crc32 := (Crc32 SHR 8)
               END;
         END
      ELSE BEGIN
            Ms_WriteChar(Chr(ZBIN));
            SZ_SendByte(htype);
            Crc16 := UpdC16(htype,0);
            FOR n := 0 TO 3 DO
               BEGIN
                  SZ_SendByte(hdr[n]);
                  Crc16 := UpdC16(hdr[n],Crc16)
               END;
            Crc16 := UpdC16(0,Crc16);
            Crc16 := UpdC16(0,Crc16);
            SZ_SendByte(Lo(Crc16 SHR 8));
            SZ_SendByte(Lo(Crc16));
         END;

   IF (htype <> ZDATA) THEN RealDelay(500)
END;


PROCEDURE SZ_SendData(Buf:pBuffer; bLength:Integer; FrameEnd:Byte);
VAR Crc16 : Word;
    Crc32 : LongInt;
    t     : Integer;
BEGIN
   Zmodem_ShowBlockSize(bLength);
   IF SendCrc32
      THEN BEGIN
            Crc32 := $FFFFFFFF;
            FOR t := 0 TO (bLength-1) DO
               BEGIN
                  SZ_SendByte(Buf^[t]);
                  Crc32 := UpdC32(Buf^[t],Crc32)
               END;
            Crc32 := UpdC32(frameend,crc32);
            Crc32 := (NOT crc32);
            Ms_WriteChar(Chr(ZDLE));
            Ms_WriteChar(Chr(FrameEnd));
            FOR t := 0 TO 3 DO
               BEGIN
                  SZ_SendByte(Byte(Crc32));
                  Crc32 := (crc32 SHR 8)
               END;
         END
      ELSE BEGIN
            Crc16 := 0;
            FOR t := 0 TO (bLength-1) DO
               BEGIN
                  SZ_SendByte(Buf^[t]);
                  Crc16 := UpdC16(Buf^[t],Crc16);
               END;
            Crc16 := UpdC16(Frameend,Crc16);
            Ms_WriteChar(Chr(ZDLE));
            Ms_WriteChar(Chr(FrameEnd));
            Crc16 := UpdC16(0,Crc16);
            Crc16 := UpdC16(0,Crc16);
            SZ_SendByte(Lo(Crc16 SHR 8));
            SZ_SendByte(Lo(Crc16));
         END;

  IF (FrameEnd = ZCRCW) THEN
     BEGIN
        Ms_WriteChar(#17);
        RealDelay(500);
     END;
END;


PROCEDURE Zmodem_EndSend;
VAR Done:Boolean;
BEGIN
   Done := FALSE;
   REPEAT
      Zmodem_PutLongIntoHeader(txpos);
      SZ_SendBinaryHeader(ZFIN,txhdr);
      CASE Zmodem_GetHeader(rxhdr) OF
         ZFIN     : BEGIN
                       Ms_WriteString('OO');
                       RealDelay(500);
                       fk_FlushOutputBuffer;
                       Exit;
                    END;
         ABORT,
         ZCAN,
         RCDO,
         ZFERR,
         TIMEDOUT : Exit;
      END;
   UNTIL Done;
END;


FUNCTION SZ_GetReceiverInfo:Integer;
VAR rxflags,n,c:INTEGER;
BEGIN
   Zmodem_Message('Synchronizing with receiver . . .');
   FOR n := 1 TO 10 DO
      BEGIN
         c := Zmodem_GetHeader(rxhdr);
         Zmodem_Frame(c);
         CASE c OF
            ZCHALLENGE : BEGIN
                            Zmodem_PutLongIntoHeader(rxpos);
                            Zmodem_SendHexHeader(ZACK,txhdr)
                         END;
            ZCOMMAND   : BEGIN
                            Zmodem_PutLongIntoHeader(LongInt(0));
                            Zmodem_SendHexHeader(ZRQINIT,txhdr)
                         END;
            ZRINIT     : BEGIN
                            rxBufLen := (Word(rxhdr[ZP1]) SHL 8) OR rxhdr[ZP0];
                            SendCrc32 := ((rxhdr[ZF0] AND CANFC32) <> 0);
                            IF SendCrc32
                               THEN Zmodem_ShowCheck('Zmodem CRC-32/'+StrFunc(ZBUFSIZE))
                               ELSE Zmodem_ShowCheck('Zmodem CRC-16/'+StrFunc(ZBUFSIZE));
                            SZ_GetReceiverInfo := ZOK;
                            Exit;
                         END;
            TIMEDOUT   : Zmodem_Message('Receiver timed out; retry #'+StrFunc(n));
            ABORT,
            ZCAN,
            RCDO       : BEGIN
                            SZ_GetReceiverInfo := c;
                            Exit;
                         END;
            ELSE         IF (c <> ZRQINIT) OR (rxhdr[ZF0] <> ZCOMMAND) THEN
                            Zmodem_SendHexHeader(ZNAK,txhdr)
         END;
      END;
   SZ_GetReceiverInfo := ZERROR;
END;


FUNCTION SZ_SyncWithReceiver:Integer;
VAR c,
    num_errs : Integer;
    NumStr   : String[10];
    Done     : Boolean;
BEGIN
   num_errs := 7;
   Done := FALSE;
   REPEAT
      c := Zmodem_GetHeader(rxhdr);
      fk_PurgeInputBuffer;
      Zmodem_Frame(c);
      CASE c OF
         TIMEDOUT : BEGIN
                       Dec(num_errs);
                       IF (num_errs < 0) THEN
                          BEGIN
                             Zmodem_Message('Receiver timed out!');
                             SZ_SyncWithReceiver := ZERROR;
                             Exit;
                          END;
                    END;
         ZABORT,
         ZCAN     : BEGIN
                       Zmodem_Message('Transfer aborted by remote!');
                       SZ_SyncWithReceiver := c;
                       Exit;
                    END;
         ABORT,
         ZFIN,
         RCDO     : BEGIN
                       Zmodem_Message('Transfer aborted!');
                       SZ_SyncWithReceiver := c;
                       Exit;
                    END;
         ZRPOS    : BEGIN
                       Seek(TheFile,RxPos);
                       IF IOResult <> 0
                          THEN BEGIN
                                Zmodem_Message('File seek error');
                                SZ_SyncWithReceiver := ZERROR;
                             END
                          ELSE BEGIN
                                Zmodem_Message(StrFunc(FileTx)+': Error, bad CRC');
                                Zmodem_Message(StrFunc(FileTx)+': Resuming from '+StrFunc(RxPos));
                                FileTx := RxPos;
                                Zmodem_ShowLoc;
                                Zmodem_ShowBatchLoc;
                                txpos := rxpos;
                                SZ_SyncWithReceiver := c;
                             END;
                       Exit;
                    END;
         ZSKIP,
         ZRINIT,
         ZACK     : BEGIN
                       Zmodem_Message('Waiting to send file');
                       SZ_SyncWithReceiver := c;
                       Exit;
                    END
         ELSE       BEGIN
                       Zmodem_Message(StrFunc(FileTx)+': Garbled header');
                       Zmodem_Message('GO LEAFS GO! :)');
                       SZ_SendBinaryHeader(ZNAK,txhdr)
                    END
      END;
   UNTIL Done;
END;


FUNCTION SZ_SendFileData:Integer;
LABEL WAITACK, SOMEMORE;
VAR   c,
      e           : Integer;
      NewCnt,
      BlkLen,
      MaxBlkLen,
      GoodBlks,
      GoodNeeded  : Word;
      Ch          : Char;
      Stop        : Boolean;
      ChFlag      : Boolean;
      CanCount    : Byte;
      {$IFDEF OS2}
      BytesRead   : LongInt;
      {$ELSE}
      BytesRead   : Integer;
      {$ENDIF}
BEGIN
   Zmodem_Message('Sending file...');
   GoodNeeded := 4;
   IF (fk_Fossil.Baud < 300)
      THEN maxblklen := 128
      ELSE maxblklen := (Word(fk_Fossil.Baud) DIV 300)*256;

   IF (maxblklen > ZBUFSIZE) THEN maxblklen := ZBUFSIZE;
   IF (rxbuflen > 0) AND (rxbuflen < maxblklen) THEN maxblklen := rxbuflen;

   BlkLen := ZBUFSIZE;

   StartTime := CurrentSecsFunc;
   fk_PurgeInputBuffer;
{   SZ_SendBinaryHeader(ZFILE,txhdr);}

SOMEMORE:

   IF fk_RemoteKeyPressed THEN
      BEGIN

WAITACK:

         c := SZ_SyncWithReceiver;
         Zmodem_Frame(c);
         CASE c OF
            ZCAN,
            ABORT,
            ZSKIP  : BEGIN
                        SZ_SendFileData := c;
                        Exit;
                     END;
            ZACK   : { null };
            ZRPOS  : BEGIN
                        Inc(Errors);
                        Zmodem_Errors(Errors);
                        IF (blklen SHR 2) > 32
                           THEN blklen := (blklen SHR 2)
                           ELSE blklen := 32;
                        goodblks := 0;
                        goodneeded := (goodneeded SHL 1) OR 1
                     END;
            ZRINIT : BEGIN
                        SZ_SendFileData := ZOK;
                        Exit;
                     END;
            ELSE     BEGIN
                        SZ_SendFileData := ZERROR;
                        Exit;
                     END;
         END;
         WHILE fk_RemoteKeyPressed DO
            CASE (MSZ_TimedRead(2)) OF
               CAN,
               ZPAD : GOTO WaitAck;
               RCDO : BEGIN
                         SZ_SendFileData := ZERROR;
                         Exit;
                      END;
            END;
      END;

   newcnt := rxbuflen;
   Zmodem_PutLongIntoHeader(txpos);
   SZ_SendBinaryHeader(ZDATA,txhdr);
   Zmodem_Message('Sending data header');

   REPEAT
      IF KeyPressed AND (ReadKey = #27) THEN
         BEGIN
            Zmodem_Message('Aborted from keyboard');
            SZ_SendFileData := ABORT;
            Exit;
         END;

      IF (NOT Ms_CheckCarrier) THEN
         BEGIN
            Zmodem_Message('Carrier lost');
            SZ_SendFileData := RCDO;
            Exit;
         END;

      BlockRead(TheFile,TxBuf^,BlkLen,BytesRead);
      IF (BytesRead < blklen)
         THEN e := ZCRCE
         ELSE IF (rxbuflen <> 0) AND ((newcnt - BytesRead) <= 0)
            THEN BEGIN
                  newcnt := (newcnt - BytesRead);
                  e := ZCRCW
               END
            ELSE e := ZCRCG;

      SZ_SendData(txbuf,BytesRead,e);
      txpos := txpos + BytesRead;
      Inc(goodblks);
      IF (blklen < maxblklen) AND (goodblks > goodneeded) THEN
         BEGIN
            IF (blklen SHL 1) < maxblklen
               THEN blklen := (blklen SHL 1)
               ELSE blklen := maxblklen;
            goodblks := 0
         END;
      FileTx := TxPos;
      Zmodem_ShowLoc;
      Zmodem_ShowBatchLoc;
      IF (e = ZCRCW) THEN GOTO WaitAck;

      CanCount := 0;
      WHILE fk_RemoteKeyPressed DO
         BEGIN
            c := MSZ_TimedRead(2);
            CASE c OF
               CAN,
               ZPAD  : BEGIN
                          Zmodem_Message('Synchronizing with receiver');
                          SZ_SendData(TxBuf,0,ZCRCE);
                          GOTO WaitAck;
                       END;
               RCDO,
               ABORT : BEGIN
                          SZ_SendFileData := c;
                          Exit;
                       END;
            END;
         END;
   UNTIL (e <> ZCRCG);

   Stop := FALSE;
   REPEAT
      Zmodem_PutLongIntoHeader(txpos);
      Zmodem_Message('Sending EOF');
      SZ_SendBinaryHeader(ZEOF,txhdr);
      c := SZ_SyncWithReceiver;
      CASE c OF
         ABORT  : BEGIN
                     SZ_SendFileData := c;
                     Stop := TRUE;
                  END;
         ZACK   : Stop := TRUE;
         ZRPOS  : GOTO SomeMore;
         ZRINIT : BEGIN
                     SZ_SendFileData := ZOK;
                     Zmodem_Message('Transfer completed');
                     Stop := TRUE;
                  END;
         ZSKIP  : BEGIN
                     SZ_SendFileData := ZOK;
                     Zmodem_Message('(?) Possible leech Zmodem');
                     Zmodem_Message('     got SKIP after EOF');
                     Exit;
                  END;
         ZCAN   : BEGIN
                     SZ_SendFileData := ZOK;
                     Zmodem_Message('(?) Possible leech Zmodem');
                     Zmodem_Message('     got CAN after EOF');
                     Exit;
                  END;
         ELSE     BEGIN
                     SZ_SendFileData := ZERROR;
                     Exit;
                  END;
      END;
      FileTx := TxPos;
      Zmodem_ShowLoc;
      Zmodem_ShowBatchLoc;
   UNTIL (c <> ZACK);
END;


FUNCTION SZ_SendFile:Integer;
VAR c:Integer; Done:Boolean;
BEGIN
   Errors := 0;
   Done := FALSE;
   REPEAT
      FillChar(txhdr,4,0);
      txhdr[ZF0] := ZCRESUM; { resume }
      Zmodem_Message('Ready to send');
      SZ_SendBinaryHeader(ZFILE,txhdr);
      SZ_SendData(txbuf,256{ZBUFSIZE},ZCRCW);
      REPEAT
         c := Zmodem_GetHeader(rxhdr);
         Zmodem_Frame(c);
         CASE c OF
            ABORT,
            ZCAN,
            TIMEDOUT,
            RCDO,
            ZFIN,
            ZABORT    : BEGIN
                           SZ_SendFile := c;
                           Exit;
                        END;
            ZRINIT    : ;
            ZCRC      : BEGIN
                           Zmodem_PutLongIntoHeader(Zmodem_FileCRC32(TheFile));
                           Zmodem_SendHexHeader(ZCRC,txhdr)
                        END;
            ZSKIP     : BEGIN
                           SZ_SendFile := c;
                           Exit;
                        END;
            ZRPOS     : BEGIN
                           Seek(TheFile,RxPos);
                           IF IOResult <> 0 THEN
                              BEGIN
                                 Zmodem_Message('File positioning error');
                                 Zmodem_SendHexHeader(ZFERR,txhdr);
                                 SZ_SendFile := ZERROR;
                                 Exit;
                              END;
                           IF (rxpos = 0)
                              THEN FileAddition := NewFile
                              ELSE FileAddition := RecoverFile;

                           Zmodem_Message('Setting start position');
                           TxPos := RxPos;
                           FileTx := TxPos;
                           FileTxStart := TxPos;
                           Zmodem_ShowLoc;
                           Zmodem_ShowBatchLoc;
                           SZ_SendFile := SZ_SendFileData;
                           Exit;
                        END
         END;
      UNTIL (c <> ZRINIT) OR Done;
   UNTIL Done;
END;


FUNCTION Zmodem_SendFile(fn:String):Boolean;
VAR s:String; n:Integer;

    FUNCTION Send_ZRQINIT:Boolean;
    BEGIN
       StartTime := $7FFFFFFF;
       Zmodem_ShowName(fn);
       Zmodem_ShowSize;

       s := DownCaseStr(
                        FileName+#0+
                        StrFunc(FileTotal)+' '+
                        Zmodem_ToUnixDate(FileTime)+' '+
                        '0 '+
                        '0 '+
                        StrFunc(BatchNum-BatchCurrent)+' '+
                        StrFunc(BatchTotal-BatchTx)
                       );

       FillChar(txbuf^,ZBUFSIZE,0);
       Move(s[1],txbuf^[0],Length(s));
       IF (fk_Fossil.Baud > 0)
          THEN rxTimeout := Integer(614400 DIV fk_Fossil.Baud)
          ELSE rxTimeout := 100;
       IF (rxTimeout < 100) THEN rxTimeout := 100;
       Attn := '';
       Zmodem_PutLongIntoHeader(LONGINT(0));
       Ms_WriteString('rz'+#13);
       Zmodem_Message('Sending ZRQINIT');
       Zmodem_SendHexHeader(ZRQINIT,txhdr);

       IF (SZ_GetReceiverInfo < 0) THEN
          BEGIN
             MSZ_WriteLog('z',0,fn,FALSE);
             Send_ZRQINIT := FALSE;
             Zmodem_SendFile := FALSE;
             Exit;
          END;

      Send_ZRQINIT := TRUE;
   END;

   FUNCTION Send_Init:Boolean;
   BEGIN
      Errors := 0;
      IF NOT Ms_CheckCarrier
         THEN BEGIN
               Zmodem_Message('Lost carrier');
               RealDelay(2000);
               Send_Init := FALSE;
               Zmodem_SendFile := FALSE;
            END
         ELSE IF NOT Zmodem_FindFile(fn,FileName,FileTotal,FileTime) THEN
            BEGIN
               Zmodem_Message('Unable to find/open file');
               Zmodem_EndSend;
               Send_Init := FALSE;
               Zmodem_SendFile := FALSE;
            END
         ELSE Send_Init := TRUE;
   END;

BEGIN
   Zmodem_InitWindow(TRUE);
   IF Send_Init THEN IF Send_ZRQINIT THEN
      BEGIN
         Assign(TheFile,fn);
         Reset(TheFile,1);
         FileTotal := FileSize(TheFile);
         StartTime := CurrentSecsFunc;

         n := SZ_SendFile;
         Close(TheFile);
         Zmodem_Frame(n);
         FileTx := TxPos;
         IF (n <> ZOK) THEN MSZ_SendCan;
         BatchTx := BatchTx+FileTotal;
         Inc(BatchCurrent);
         MSZ_WriteLog('z',txPos,fn,n = ZOK);
         Zmodem_SendFile := n IN [ZOK,ZSKIP];
      END;
END;


FUNCTION Zmodem_Send(FileSpec:String):Boolean;
VAR FileNum:Integer; DirInfo:SearchRec; FileName:String;
BEGIN
   FileNum := 1;
   FFirst(FileSpec,AnyFile AND NOT VolumeID,DirInfo);
   IF DosError <> 0
      THEN Zmodem_Send := FALSE
      ELSE BEGIN
            WHILE DosError = 0 DO
               BEGIN
                  FileName := AddSlash(ExtractFDir(FileSpec))+DirInfo.Name;
                  IF NOT Zmodem_SendFile(DowncaseStr(FileName)) THEN
                     BEGIN
                        Zmodem_Send := FALSE;
                        {$IFDEF DELPHI} SYSUTILS.FindClose(DirInfo); {$ENDIF}
                        {$IFDEF OS2}    FindClose(DirInfo); {$ENDIF}
                        Exit;
                     END;
                  Inc(FileNum);
                  FNext(DirInfo);
               END;
            {$IFDEF DELPHI} SYSUTILS.FindClose(DirInfo); {$ENDIF}
            {$IFDEF OS2}    FindClose(DirInfo); {$ENDIF}
            Zmodem_Send := TRUE;
         END;
END;


END.



