PROGRAM SetupCfg;

{ this program creates a new WaterGate config by creating an empty }
{ database and reading from config.txt for commands to fill the    }
{ database. It directly points to the current source directories   }
{ to get the latest structures.                                    }

{$IFNDEF WtrConf} ### Need define WtrConf ### {$ENDIF}

USES Ramon,
     Dos,
     Cfg,
     Database,
     Tdb,
     Fido,
     UserBase,
     AreaBase,
     FlexCfg;

VAR WorkDrive     : STRING[10];
    LogFile       : TEXT;
    FlexPos       : LONGINT;
    Filter        : FilterRecord;
    AreaFixFwd    : AreaFixForwardRecord;
    ListRec       : MailingListRecord;
    ListPos       : LONGINT;
    ListUserRec   : ListUserRecord;
    ListUserRecNr : ListUserRecordNrType;

FUNCTION TimeStamp : STRING;

VAR Hour,Min,Sec,SecH  : WordLong;

BEGIN
     GetTime (Hour,Min,Sec,SecH);

     TimeStamp:=AddUpWithPre0s (2,Word2String (Hour))+':'+
                AddUpWithPre0s (2,Word2String (Min))+':'+
                AddUpWithPre0s (2,Word2String (Sec));
END;


PROCEDURE Log (Msg : STRING);
BEGIN
     Msg:=TimeStamp+Spaces (4)+Msg;
     WriteLn (LogFile,Msg);
     WriteLn (Msg);
END;


PROCEDURE LogErr (IORes : BYTE; Msg : STRING);
BEGIN
     Log (Msg+' (error '+Byte2String (IORes)+')');
END;


PROCEDURE StartSetup (SystemPath : STRING);
BEGIN
     CreatePath (CorrectPath (SystemPath));

     SetupEmptyConfig;

     Config.LogDebug:=TRUE;
     Config.EmptyTear:=TRUE; { avoid trouble with news/echomail }
     Config.GateMsgId:=gmInclude;
     Config.AlwaysMimeQuotePrint:=FALSE;

     Config.SystemDir:=CorrectPath (SystemPath);
     OpenDatabases;

     Flex_Init;
END;


FUNCTION ReplaceToken (Regel,Token,NewText : STRING) : STRING;

VAR P : BYTE;

BEGIN
     REPEAT
           P:=Pos (Token,Regel);
           IF (P > 0) THEN
              Regel:=Copy (Regel,1,P-1)+NewText+Copy (Regel,P+Length (Token),255);
     UNTIL (P = 0);

     ReplaceToken:=Regel;
END;

PROCEDURE AddSystemDomain (Regel : STRING);

VAR Lp : 1..MaxSystemDomains;

BEGIN
     FOR Lp:=1 TO MaxSystemDomains DO
         IF (Config.Domains[Lp] = '') THEN
         BEGIN
              Config.Domains[Lp]:=Regel;
              Exit;
         END;

     Log ('SetupCfg: Too many system domains; cannot add '+Regel);
     Exit;
END;

PROCEDURE AddSystemAka (Regel : STRING);

VAR Lp : 1..MaxAkas;

BEGIN
     FOR Lp:=1 TO MaxAkas DO
         IF FidoCompare (Config.NodeNrs[Lp],NullAdres) THEN
         BEGIN
              FidoSplit (Regel,Config.NodeNrs[Lp]);
              Exit;
         END;
END;

FUNCTION Regel2MsgBaseType (Regel : STRING) : FidoMsgStyleType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'MSG') THEN
     BEGIN
          Regel2MsgBaseType:=FidoMsgType;
          Exit;
     END;

     IF (Regel = 'JAM') THEN
     BEGIN
          Regel2MsgBaseType:=JamType;
          Exit;
     END;

     IF (Regel = 'SQUISH') THEN
     BEGIN
          Regel2MsgBaseType:=SquishType;
          Exit;
     END;

     (*
     IF (Regel = 'WILDCAT') THEN
     BEGIN
          Regel2MsgBaseType:=WildCatType;
          Exit;
     END;
     *)

     Log ('SetupCfg: Unknown message base type "'+Regel+'"');

     Regel2MsgBaseType:=NoneType;
END;


FUNCTION Regel2OrigInOrigin (Regel : STRING) : BYTE;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'NO') THEN
     BEGIN
          Regel2OrigInOrigin:=0;
          Exit;
     END;

     IF (Regel = 'YES') THEN
     BEGIN
          Regel2OrigInOrigin:=1;
          Exit;
     END;

     IF (Regel = 'OVERRIDE') THEN
     BEGIN
          Regel2OrigInOrigin:=2;
          Exit;
     END;

     Log ('SetupCfg: Unknown Organization-In-Origin option "'+Regel+'"');

     Regel2OrigInOrigin:=0;
END;


FUNCTION Regel2UserType (Regel : STRING) : SystemType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = '_F') THEN
     BEGIN
          Regel2UserType:=_F;
          Exit;
     END;

     IF (Regel = '_U') THEN
     BEGIN
          Regel2UserType:=_U;
          Exit;
     END;

     IF (Regel = '_S') THEN
     BEGIN
          Regel2UserType:=_S;
          Exit;
     END;

     IF (Regel = '_BBS') THEN
     BEGIN
          Regel2UserType:=_BBS;
          Exit;
     END;

     IF (Regel = '_P') THEN
     BEGIN
          Regel2UserType:=_P;
          Exit;
     END;

     IF (Regel = '_B') THEN
     BEGIN
          Regel2UserType:=_B;
          Exit;
     END;

     Log ('SetupCfg: Unknown user type "'+Regel+'"');

     Regel2UserType:=_F;
END;


FUNCTION Regel2AreaType (Regel : STRING) : AreaRecordType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'ECHO') THEN
     BEGIN
          Regel2AreaType:=Area_Echo;
          Exit;
     END;

     IF (Regel = 'NETMAIL') THEN
     BEGIN
          Regel2AreaType:=Area_Netmail;
          Exit;
     END;

     IF (Regel = 'LOCAL') THEN
     BEGIN
          Regel2AreaType:=Area_Local;
          Exit;
     END;

     IF (Regel = 'E-MAIL') THEN
     BEGIN
          Regel2AreaType:=Area_Email;
          Exit;
     END;

     Log ('SetupCfg: Unknown area type "'+Regel+'"');

     Regel2AreaType:=Area_Echo;
END;

FUNCTION FindArea (Regel : STRING) : AreaBaseRecordNrType;

VAR Lp : AreaBaseRecordNrType;

BEGIN
     FindArea:=NILRecordNr;

     FOR Lp:=1 TO AreaBaseRecCount DO
     BEGIN
          ReadAreaBaseRecord (Lp,AreaData);

          IF CaselessMatch_CC (AreaData.AreaName_F,Regel) OR
             CaselessMatch_CC (AreaData.AreaName_U,Regel) THEN
          BEGIN
               FindArea:=Lp;
               Exit;
          END;
     END; { for }
END;

PROCEDURE Regel2Groups (Regel : STRING; VAR Groups : GroupFlagType);

VAR P  : BYTE;
    Gr : STRING[2];
    Nr : 1..MaxGroups;

BEGIN
     WHILE (Regel <> '') DO
     BEGIN
          Regel:=DeleteFrontSpaces (Regel);
          P:=Pos (' ',Regel);

          IF (P > 0) THEN
          BEGIN
               Gr:=Copy (Regel,1,P-1);
               Delete (Regel,1,P);
          END ELSE
          BEGIN
               Gr:=Regel;
               Regel:='';
          END;

          Gr[1]:=UpCase (Gr[1]);

          IF (NOT (Gr[1] IN ['A'..'Z']) AND (Gr[2] IN ['0'..'5'])) THEN
          BEGIN
               Log ('SetupCfg: Invalid group code: '+Gr);
               Exit;
          END;

          Nr:=1 + (Ord (Gr[1])-Ord ('A'))*5 + (Ord (Gr[2])-Ord ('1'));

          IF (Nr > MaxGroups) THEN
          BEGIN
               Log ('SetupCfg: Invalid group code: '+Gr+' ('+Byte2String (Nr)+')');
               Exit;
          END;

          AddGroupToGroupList (Groups,Nr);
     END; { while }
END;

FUNCTION Regel2Compression (Regel : STRING) : CompressionType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'ARC') THEN
     BEGIN
          Regel2Compression:=fctARC;
          Exit;
     END;

     IF (Regel = 'ARJ') THEN
     BEGIN
          Regel2Compression:=fctARJ;
          Exit;
     END;

     IF (Regel = 'LZH') THEN
     BEGIN
          Regel2Compression:=fctLZH;
          Exit;
     END;

     IF (Regel = 'PAK') THEN
     BEGIN
          Regel2Compression:=fctPAK;
          Exit;
     END;

     IF (Regel = 'ZIP') THEN
     BEGIN
          Regel2Compression:=fctZIP;
          Exit;
     END;

     IF (Regel = 'ZOO') THEN
     BEGIN
          Regel2Compression:=fctZOO;
          Exit;
     END;

     IF (Regel = 'RAR') THEN
     BEGIN
          Regel2Compression:=fctRAR;
          Exit;
     END;

     IF (Regel = 'OP1') THEN
     BEGIN
          Regel2Compression:=fctOP1;
          Exit;
     END;

     IF (Regel = 'NONE') THEN
     BEGIN
          Regel2Compression:=fctNone;
          Exit;
     END;

     Log ('Unknown Compression type: '+Regel);
     Regel2Compression:=fctNone;
END;

FUNCTION Regel2FidoSystem (Regel : STRING) : FidoSystemType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'BINKLEY') THEN
     BEGIN
          Regel2FidoSystem:=stBinkley;
          Exit;
     END;

     IF (Regel = 'FRONTDOOR') OR (Regel = 'FD') THEN
     BEGIN
          Regel2FidoSystem:=stFrontDoor;
          Exit;
     END;

     IF (Regel = 'DBRIDGE') THEN
     BEGIN
          Regel2FidoSystem:=stDbridge;
          Exit;
     END;

     Log ('Unknown FidoSystem: '+Regel);
     Regel2FidoSystem:=stFrontDoor;
END;

FUNCTION Regel2PktFormat (Regel : STRING) : PktFormatType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'PKT') THEN
     BEGIN
          Regel2PktFormat:=fptPkt;
          Exit;
     END;

     IF (Regel = 'P2K') OR (Regel = 'PKT2000') THEN
     BEGIN
          Regel2PktFormat:=fptPkt2000;
          Exit;
     END;

     Log ('Unknown PktFormat: '+Regel);
     Regel2PktFormat:=fptPkt;
END;

FUNCTION Regel2FilterAction (Regel : STRING) : MapActionType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'MOVE') THEN
     BEGIN
          Regel2FilterAction:=matMove;
          Exit;
     END;

     IF (Regel = 'COPY') THEN
     BEGIN
          Regel2FilterAction:=matCopy;
          Exit;
     END;

     IF (Regel = 'DELETE') THEN
     BEGIN
          Regel2FilterAction:=matDelete;
          Exit;
     END;

     IF (Regel = 'FORWARD') THEN
     BEGIN
          Regel2FilterAction:=matForward;
          Exit;
     END;

     IF (Regel = 'CC') THEN
     BEGIN
          Regel2FilterAction:=matCC;
          Exit;
     END;

     IF (Regel = 'BOUNCE') THEN
     BEGIN
          Regel2FilterAction:=matBounce;
          Exit;
     END;

     IF (Regel = 'BOUNCETO') THEN
     BEGIN
          Regel2FilterAction:=matBounceAddr;
          Exit;
     END;

     IF (Regel = 'SAVE') THEN
     BEGIN
          Regel2FilterAction:=matSave;
          Exit;
     END;

     IF (Regel = 'NONE') THEN
     BEGIN
          Regel2FilterAction:=matNone;
          Exit;
     END;

     Log ('Unknown FilterAction: '+Regel);
     Regel2FilterAction:=matMove;
END;

FUNCTION Regel2FilterWhere (Regel : STRING) : MapSearchWhereType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'ALL') THEN
     BEGIN
          Regel2FilterWhere:=mwtAll;
          Exit;
     END;

     IF (Regel = 'BODY') THEN
     BEGIN
          Regel2FilterWhere:=mwtBody;
          Exit;
     END;

     IF (Regel = 'HEADER') THEN
     BEGIN
          Regel2FilterWhere:=mwtHeader;
          Exit;
     END;

     IF (Regel = 'FROM') THEN
     BEGIN
          Regel2FilterWhere:=mwtFrom;
          Exit;
     END;

     IF (Regel = 'TO') THEN
     BEGIN
          Regel2FilterWhere:=mwtTo;
          Exit;
     END;

     IF (Regel = 'SUBJECT') THEN
     BEGIN
          Regel2FilterWhere:=mwtSubject;
          Exit;
     END;

     Log ('Unknown FilterWhere option: '+Regel);
     Regel2FilterWhere:=mwtAll;
END;

FUNCTION Regel2FilterAddrType (Regel : STRING) : MapAddressType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'FTN') THEN
     BEGIN
          Regel2FilterAddrType:=madFTN;
          Exit;
     END;

     IF (Regel = 'RFC') THEN
     BEGIN
          Regel2FilterAddrType:=madRFC;
          Exit;
     END;

     IF (Regel = 'REMOTE-GW') THEN
     BEGIN
          Regel2FilterAddrType:=madRemoteGW;
          Exit;
     END;

     Log ('Unknown FilterAddrType option: '+Regel);
     Regel2FilterAddrType:=madRFC;
END;

FUNCTION Regel2FilterLimit (Regel : STRING) : BYTE;

VAR Result : BYTE;
    Part   : STRING;
    P      : BYTE;

BEGIN
     Regel:=DeleteFrontAndBackSpaces (Regel);
     Regel:=UpCaseString (Regel);

     Result:=0;

     WHILE (Regel <> '') DO
     BEGIN
          P:=Pos (' ',Regel);
          IF (P > 0) THEN
          BEGIN
               Part:=Copy (Regel,1,P-1);
               Delete (Regel,1,P);
               Regel:=DeleteFrontSpaces (Regel);
          END ELSE
          BEGIN
               Part:=Regel;
               Regel:='';
          END;

          IF (Part = 'NETMAIL') THEN
             Result:=Result OR mltNetmail
          ELSE
              IF (Part = 'ECHOMAIL') THEN
                 Result:=Result OR mltEchomail
              ELSE
                  IF (Part = 'E-MAIL') THEN
                     Result:=Result OR mltEmail
                  ELSE
                      IF (Part = 'NEWS') THEN
                         Result:=Result OR mltNews
                      ELSE
                          Log ('Unknown FilterLimit option: '+Part);
     END; { while }

     Regel2FilterLimit:=Result;
END;

FUNCTION Regel2FilterInOut (Regel : STRING) : BYTE;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'INBOUND') THEN
     BEGIN
          Regel2FilterInOut:=mioInbound;
          Exit;
     END;

     IF (Regel = 'OUTBOUND') THEN
     BEGIN
          Regel2FilterInOut:=mioOutbound;
          Exit;
     END;

     IF (Regel = 'BOTH') THEN
     BEGIN
          Regel2FilterInOut:=mioBoth;
          Exit;
     END;

     Log ('Unknown FilterInOut option: '+Regel);
     Regel2FilterInOut:=mioBoth;
END;

FUNCTION Regel2GateMsgIdType (Regel : STRING) : GateMsgIdType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'NOT') THEN
     BEGIN
          Regel2GateMsgIdType:=gmNot;
          Exit;
     END;

     IF (Regel = 'INCLUDE') THEN
     BEGIN
          Regel2GateMsgIdType:=gmInclude;
          Exit;
     END;

     Log ('Unknown GateMsgID option: '+Regel);
     Regel2GateMsgIdType:=gmInclude;
END;

FUNCTION RegelYesNo2Boolean (Keyword : STRING; Regel : STRING) : BOOLEAN;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'YES') THEN
     BEGIN
          RegelYesNo2Boolean:=TRUE;
          Exit;
     END;

     IF (Regel = 'NO') THEN
     BEGIN
          RegelYesNo2Boolean:=FALSE;
          Exit;
     END;

     Log ('Unknown '+Keyword+' option: '+Regel);
     RegelYesNo2Boolean:=FALSE;
END;

FUNCTION Regel2ListUserType (Regel : STRING) : ListUserType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'RFC') THEN
     BEGIN
          Regel2ListUserType:=lutRFC;
          Exit;
     END;

     IF (Regel = 'FTN') THEN
     BEGIN
          Regel2ListUserType:=lutFTN;
          Exit;
     END;

     IF (Regel = 'REMOTEGW') THEN
     BEGIN
          Regel2ListUserType:=lutRemoteGW;
          Exit;
     END;

     IF (Regel = 'DELETED') THEN
     BEGIN
          Regel2ListUserType:=lutDeleted;
          Exit;
     END;

     Log ('Unknown ListUserType option: '+Regel);
     Regel2ListUserType:=lutDeleted;
END;

FUNCTION Regel2AreaFixFwdListType (Regel : STRING) : AreaListType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'AREASBBS') THEN
     BEGIN
          Regel2AreaFixFwdListType:=lstAREASBBS;
          Exit;
     END;

     IF (Regel = 'NAMES') THEN
     BEGIN
          Regel2AreaFixFwdListType:=lstNAMES;
          Exit;
     END;

     Log ('Unknown AreaFixFwdList option: '+Regel);
     Regel2AreaFixFwdListType:=lstNAMES;
END;

FUNCTION Regel2OneGroupNr (Regel : STRING) : BYTE;

VAR C  : CHAR;
    CN : BYTE;

BEGIN
     Regel2OneGroupNr:=1;

     IF (Length (Regel) <> 2) THEN
     BEGIN
          Log ('Group must be formatted like A2, not "'+Regel+'"');
          Exit;
     END;

     C:=UpCase (Regel[1]);
     IF (NOT (C IN ['A'..'Z'])) THEN
     BEGIN
          Log ('Invalid group letter in "'+Regel+'"');
          Exit;
     END;

     C:=Regel[2];
     IF (NOT (C IN ['1'..'5'])) THEN
     BEGIN
          Log ('Invalid group number in "'+Regel+'"');
          Exit;
     END;

     CN:=Ord (C)-Ord ('0');

     Regel2OneGroupNr:=(Ord (UpCase (Regel[1]))-Ord ('A'))*5+CN;
END;

FUNCTION Regel2ListPriority (Regel : STRING) : ListAddrType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'HIGHER') THEN
     BEGIN
          Regel2ListPriority:=laLower;
          Exit;
     END;

     IF (Regel = 'LOWER') THEN
     BEGIN
          Regel2ListPriority:=laHigher;
          Exit;
     END;

     Log ('Unknown ListPriority option: '+Regel);
     Regel2ListPriority:=laHigher;
END;

FUNCTION Regel2ListAccess (Regel : STRING) : ListAccessType;
BEGIN
     Regel:=UpCaseString (Regel);

     IF (Regel = 'READWRITE') OR (Regel = 'RW') THEN
     BEGIN
          Regel2ListAccess:=laReadWrite;
          Exit;
     END;

     IF (Regel = 'READONLY') OR (Regel = 'RO') THEN
     BEGIN
          Regel2ListAccess:=laReadOnly;
          Exit;
     END;

     IF (Regel = 'WRITEONLY') OR (Regel = 'WO') THEN
     BEGIN
          Regel2ListAccess:=laWriteOnly;
          Exit;
     END;

     Log ('Unknown ListDefaultAccess option: '+Regel);
     Regel2ListAccess:=laReadWrite;
END;

PROCEDURE ReadConfigTxt (Path : STRING);

VAR InFile    : TEXT;
    IORes     : BYTE;
    Keyword,
    Regel     : STRING;
    LineNr    : WORD;
    P         : BYTE;
    FirstSU   : BOOLEAN;
    Lp        : BYTE;
    AreaRecNr : AreaBaseRecordNrType;
    FakeAka   : FidoAddrType;
    Nop       : ValNop;

BEGIN
     Assign (InFile,Path);
     {$I-} Reset (InFile); {$I+} IORes:=IOResult;
     IF (IORes <> 0) THEN
     BEGIN
          LogErr (IORes,'Error opening '+Path);
          Exit;
     END;

     LineNr:=0;
     FirstSU:=TRUE;

     WHILE (NOT Eof (InFile)) DO
     BEGIN
          ReadLn (InFile,Regel);
          Inc (LineNr);

          Regel:=DeleteFrontAndBackSpaces (CleanTabs (Regel,1));

          IF (Regel = '') OR (Regel[1] = ';') THEN
             Continue;

          { replace all tokens }
          Regel:=ReplaceToken (Regel,'$:',WorkDrive);

          P:=Pos (' ',Regel);
          IF (P > 0) THEN
          BEGIN
               Keyword:=UpCaseString (Copy (Regel,1,P-1));
               Delete (Regel,1,P);
               Regel:=DeleteFrontSpaces (Regel);
          END ELSE
          BEGIN
               Keyword:=UpCaseString (Regel);
               Regel:='';
          END;

          IF (Keyword = 'SYSTEMDIR') THEN
          BEGIN
               IF (NOT FirstSU) THEN
               BEGIN
                    Log ('SetupCfg: Only one SystemDir keyword allowed');
                    Exit;
               END;
               FirstSU:=FALSE;

               StartSetup (Regel);
               Continue;
          END;

          IF FirstSU THEN
          BEGIN
               Log ('SetupCfg: Need SystemDir before keyword '+Keyword+' in line '+Word2String (LineNr));
               Exit;
          END;

          IF (Keyword = 'FIDOSYSTEM') THEN
          BEGIn
               Config.FidoSystem:=Regel2FidoSystem (Regel);
               Continue;
          END;

          IF (Keyword = 'MAXFIDOMSGLEN') THEN
          BEGIN
               Val (Regel,Config.MaxFidoMsgLen,Nop);
               Continue;
          END;

          IF (Keyword = 'SPOOLDIR') THEN
          BEGIN
               CreatePath (CorrectPath (Regel));
               Config.SpoolBaseDir:=Regel;
               Continue;
          END;

          IF (Keyword = 'SYSTEMDOMAIN') THEN
          BEGIN
               AddSystemDomain (Regel);
               Continue;
          END;

          IF (Keyword = 'SYSTEMAKA') THEN
          BEGIN
               AddSystemAka (Regel);
               Continue;
          END;

          IF (Keyword = 'SYSTEMNETMAILTYPE') THEN
          BEGIN
               Config.FidoNetmailType:=Regel2MsgBaseType (Regel);
               Continue;
          END;

          IF (Keyword = 'SYSTEMNETMAILPATH') THEN
          BEGIN
               CreatePath (CorrectPath (Regel));
               Config.FidoNetmailPath:=Regel;
               Continue;
          END;

          IF (Keyword = 'SYSTEMBADTYPE') THEN
          BEGIN
               Config.FidoBadAreaType:=Regel2MsgBaseType (Regel);
               Continue;
          END;

          IF (Keyword = 'ORGANIZATION-IN-ORIGIN') THEN
          BEGIN
               Config.OrganizationInOrigin:=Regel2OrigInOrigin (Regel);
               Continue;
          END;

          IF (Keyword = 'SYSTEMBADPATH') THEN
          BEGIN
               CreatePath (CorrectPath (Regel));
               Config.FidoBadPath:=Regel;
               Continue;
          END;

          IF (Keyword = 'SMARTHOST') THEN
          BEGIN
               Config.SmartHost:=Regel;
               Continue;
          END;

          IF (Keyword = 'INBOUND1') THEN
          BEGIN
               CreatePath (CorrectPath (Regel));
               Config.Inbound_F[1]:=Regel;
               Continue;
          END;

          IF (Keyword = 'INBOUND2') THEN
          BEGIN
               CreatePath (CorrectPath (Regel));
               Config.Inbound_F[2]:=Regel;
               Continue;
          END;

          IF (Keyword = 'INBOUND1SECURE') THEN
          BEGIN
               Config.Security_F[1]:=Secure;
               Continue;
          END;

          IF (Keyword = 'INBOUND2SECURE') THEN
          BEGIN
               Config.Security_F[2]:=Secure;
               Continue;
          END;

          IF (Keyword = 'OUTBOUND') THEN
          BEGIN
               CreatePath (CorrectPath (Regel));
               Config.Outbound_F:=Regel;
               Continue;
          END;

          IF (Keyword = 'HEADERFULLNAME') THEN
          BEGIN
               Config.HeaderFullName:=RegelYesNo2Boolean ('HeaderFullName',Regel);
               Continue;
          END;

          IF (Keyword = 'GATEMSGID') THEN
          BEGIN
               Config.GateMsgID:=Regel2GateMsgIdType (Regel);
               Continue;
          END;

          IF (Keyword = 'DEFAULTCOMPRESSION') THEN
          BEGIN
               Config.DefaultCompressor:=Regel2Compression (Regel);
               Continue;
          END;

          IF (Keyword = 'NEWUSER') OR (Keyword = 'NEWLINK') THEN
          BEGIN
               UserData.System:=_F;

               IF CaselessMatch (Regel,'RFC') OR
                  CaselessMatch (Regel,'UUCP') THEN
                  UserData.System:=_U;

               IF CaselessMatch (Regel,'FTN') THEN
                   UserData.System:=_F;

               IF CaselessMatch (Regel,'SMTP') THEN
                  UserData.System:=_S;

               IF CaselessMatch (Regel,'BBS') THEN
                  UserData.System:=_BBS;

               IF CaselessMatch (Regel,'POP3') THEN
                  UserData.System:=_P;

               IF CaselessMatch (Regel,'SOUP') THEN
                  UserData.System:=_SOUP;

               IF CaselessMatch (Regel,'BAG') THEN
                  UserData.System:=_B;

               EmptyUserDataRecord;
               CleanEdittedUserRecord;

               { write it now so we can start adding areas }
               WriteUserBaseRecord (UserBaseRecCount+1,UserData);
               Continue;
          END;

          IF (Keyword = 'WRITEUSER') OR (Keyword = 'WRITELINK') THEN
          BEGIN
               WriteUserBaseRecord (UserBaseRecCount,UserData);
               Continue;
          END;

          IF (Keyword = 'ADDRESS') THEN
          BEGIN
               FidoSplit (Regel,UserData.Address);
               Continue;
          END;

          IF (Keyword = 'USERTYPE') THEN
          BEGIN
               UserData.System:=Regel2UserType (Regel);
               Continue;
          END;

          IF (Keyword = 'UUCPNAME') THEN
          BEGIN
               CreatePath (CorrectPath (Config.SpoolBaseDir)+Regel);
               UserData.UucpName:=Regel;
               Continue;
          END;

          IF (Keyword = 'SYSTEMNAME') THEN
          BEGIN
               UserData.UucpName:=Regel;
               Continue;
          END;

          IF (Keyword = 'SMTPINPATH') THEN
          BEGIN
               CreatePath (CorrectPath (Regel));
               UserData.SmtpInPath:=Regel;
               Continue;
          END;

          IF (Keyword = 'SMTPOUTPATH') THEN
          BEGIN
               CreatePath (CorrectPath (Regel));
               UserData.SmtpOutPath:=Regel;
               Continue;
          END;

          IF (Keyword = 'DOMAIN') THEN
          BEGIN
               FOR Lp:=1 TO MaxUserDomains DO
                   IF (UserData.Domains[Lp] = '') THEN
                   BEGIN
                        UserData.Domains[Lp]:=Regel;
                        Break;
                   END;
               Continue;
          END;

          IF (Keyword = 'COMPRESSION') THEN
          BEGIN
               UserData.Compression:=Regel2Compression (Regel);
               Continue;
          END;

          IF (Keyword = 'PACKETPWD') THEN
          BEGIN
               UserData.PacketPwd:=Regel;
               Continue;
          END;

          IF (Keyword = 'AREAFIXPWD') OR (Keyword = 'NEWSFIXPWD') THEN
          BEGIN
               UserData.AreaFixPwd:=Regel;
               Continue;
          END;

          IF (Keyword = 'PKTFORMAT') THEN
          BEGIN
               UserData.PktFormat:=Regel2PktFormat (Regel);
               Continue;
          END;

          IF (Keyword = 'BBSINBOUND') THEN
          BEGIN
               CreatePath (CorrectPath (Regel));
               UserData.Inbound:=Regel;
               Continue;
          END;

          IF (Keyword = 'BBSOUTBOUND') THEN
          BEGIN
               CreatePath (CorrectPath (Regel));
               UserData.Outbound:=Regel;
               Continue;
          END;

          IF (Keyword = 'BBSFAKEAKA') THEN
          BEGIN
               FidoSplit (Regel,FakeAka);
               UserData.FakeZone:=FakeAka.Zone;
               UserData.FakeNet:=FakeAka.Net;
               UserData.FakeNode:=FakeAka.Node;
               Continue;
          END;

          IF (Keyword = 'BBSINBOUNDEXT') THEN
          BEGIN
               UserData.InboundExt:=Regel;
               Continue;
          END;

          IF (Keyword = 'BBSKEEPSBP') THEN
          BEGIN
               UserData.KeepSBP:=RegelYesNo2Boolean ('BbsKeepSBP',Regel);
               Continue;
          END;

          {
          IF (Keyword = 'BBSSYSTEMAKA') THEN
          BEGIN
               UserData.SystemAKA    : BYTE; (1..MaxAkas)
               Continue;
          END;
          }

          IF (Keyword = 'POP3FILE') THEN
          BEGIN
               CreatePath (CorrectPath (Regel));
               UserData.Pop3File:=Regel;
               Continue;
          END;

          IF (Keyword = 'POP3RECIPIENT') THEN
          BEGIN
               UserData.Pop3Recipient:=Regel;
               Continue;
          END;

          IF (Keyword = 'POP3SEPARATOR') THEN
          BEGIN
               UserData.Pop3Separator:=Regel;
               Continue;
          END;

          IF (Keyword = 'POP3ENVELOPEHEADER') THEN
          BEGIN
               UserData.Pop3EnvelopeHdr:=Regel;
               Continue;
          END;

          IF (Keyword = 'BAGBACKLINK') THEN
          BEGIN
               UserData.BAGbacklink:=Regel;
               Continue;
          END;

          IF (Keyword = 'BAGPATH') THEN
          BEGIN
               CreatePath (CorrectPath (Regel));
               UserData.BagPath:=Regel;
               Continue;
          END;

          IF (Keyword = 'NEWAREA') THEN
          BEGIN
               EmptyAreaDataRecord (AreaData);
               WriteNewAreaBaseRecord (AreaData);
               Continue;
          END;

          IF (Keyword = 'WRITEAREA') THEN
          BEGIN
               WriteAreaBaseRecord (AreaBaseRecCount,AreaData);
               Continue;
          END;

          IF (Keyword = 'FTNNAME') THEN
          BEGIN
               AreaData.AreaName_F:=Regel;
               Continue;
          END;

          IF (Keyword = 'RFCNAME') THEN
          BEGIN
               AreaData.AreaName_U:=Regel;
               Continue;
          END;

          IF (Keyword = 'AREATYPE') THEN
          BEGIN
               AreaData.AreaType:=Regel2AreaType (Regel);
               Continue;
          END;

          IF (Keyword = 'MANDATORY') THEN
          BEGIN
               AreaData.Mandatory:=RegelYesNo2Boolean ('Mandatory',Regel);
               Continue;
          END;

          IF (Keyword = 'ALLOWPASSIVE') THEN
          BEGIN
               AreaData.AllowPassive:=RegelYesNo2Boolean ('AllowPassive',Regel);
               Continue;
          END;

          IF (Keyword = 'PASSIVE') THEN
          BEGIN
               AreaData.Passive:=RegelYesNo2Boolean ('Passive',Regel);
               Continue;
          END;

          IF (Keyword = 'MSGBASETYPE') THEN
          BEGIN
               AreaData.FidoMsgStyle:=Regel2MsgBaseType (Regel);
               Continue;
          END;

          IF (Keyword = 'MSGBASEPATH') THEN
          BEGIN
               CreatePath (CorrectPath (Regel));
               AreaData.FidoMsgPath:=Regel;
               Continue;
          END;

          IF (Keyword = 'AREAGROUPS') THEN
          BEGIN
               Regel2Groups (Regel,AreaData.IsInGroups);
               Continue;
          END;

          IF (Keyword = 'USERGROUPS') THEN
          BEGIN
               Regel2Groups (Regel,UserData.Groups);
               Continue;
          END;

          IF (Keyword = 'SUBSCRIBE') THEN
          BEGIN
               { subscribe the current user to an area }
               WriteUserBaseRecord (UserBaseRecCount,UserData);

               AreaRecNr:=FindArea (Regel);
               IF (AreaRecNr <> NILRecordNr) THEN
               BEGIN
                    ReadAreaBaseRecord (AreaRecNr,AreaData);
                    AddUserToAreaSubscrList (AreaData,UserBaseRecCount);
                    WriteAreaBaseRecord (AreaRecNr,AreaData);

                    AddAreaToUserSubscrToList (UserData,AreaRecNr);
                    WriteUserBaseRecord (UserBaseRecCount,UserData);
               END ELSE
                   Log ('SetupCfg: Cannot subscribe unknown area '+Regel);

               Continue;
          END;

          IF (Keyword = 'NEWFILTER') THEN
          BEGIN
               Flex_Filter_New (FlexPos,Filter);
               Continue;
          END;

          IF (Keyword = 'FILTERACTION') THEN
          BEGIN
               Filter.Action:=Regel2FilterAction (Regel);
               Continue;
          END;

          IF (Keyword = 'FILTERSEARCH') THEN
          BEGIN
               Filter.Search:=Regel;
               Continue;
          END;

          IF (Keyword = 'FILTERLIMIT') THEN
          BEGIN
               Filter.Limit:=Regel2FilterLimit (Regel);
               Continue;
          END;

          IF (Keyword = 'FILTERINOUT') THEN
          BEGIN
               Filter.InOut:=Regel2FilterInOut (Regel);
               Continue;
          END;

          IF (Keyword = 'FILTERWHERE') THEN
          BEGIN
               Filter.Where:=Regel2FilterWhere (Regel);
               Continue;
          END;

          IF (Keyword = 'FILTERNOTE') THEN
          BEGIN
               Filter.Note:=RegelYesNo2Boolean ('FilterNote',Regel);
               Continue;
          END;

          IF (Keyword = 'FILTERADDRTYPE') THEN
          BEGIN
               Filter.AddrType:=Regel2FilterAddrType (Regel);
               Continue;
          END;

          IF (Keyword = 'FILTERTOAKA') THEN
          BEGIN
               FidoSplit (Regel,Filter.ToAka);
               Continue;
          END;

          IF (Keyword = 'FILTERTONAME') THEN
          BEGIN
               Filter.ToName:=Regel;
               Continue;
          END;

          IF (Keyword = 'FILTERTOEMAIL') THEN
          BEGIN
               Filter.ToEmail:=Regel;
               Continue;
          END;

          IF (Keyword = 'FILTERARGUMENT') THEN
          BEGIN
               Filter.Argument:=Regel;
               Continue;
          END;

          IF (Keyword = 'WRITEFILTER') THEN
          BEGIN
               Flex_Filter_Write (FlexPos,Filter);
               Continue;
          END;

          IF (Keyword = 'NEWLIST') THEN
          BEGIN
               Flex_MailingList_New (ListPos,ListRec);
               Continue;
          END;

          IF (Keyword = 'LISTNAME') THEN
          BEGIN
               ListRec.Name:=Regel;
               Continue;
          END;

          IF (Keyword = 'LISTDESCR') THEN
          BEGIN
               ListRec.Description:=Regel;
               Continue;
          END;

          IF (Keyword = 'LISTWELCOMEFILE') THEN
          BEGIN
               ListRec.WelcomeFile:=Regel;
               Continue;
          END;

          IF (Keyword = 'LISTFOOTERFILE') THEN
          BEGIN
               ListRec.FooterFile:=Regel;
               Continue;
          END;

          IF (Keyword = 'LISTPRIVATE') THEN
          BEGIN
               ListRec.PrivateList:=RegelYesNo2Boolean ('ListPrivate',Regel);
               Continue;
          END;

          IF (Keyword = 'LISTONLYKNOWN') THEN
          BEGIN
               ListRec.OnlyKnown:=RegelYesNo2Boolean ('ListOnlyKnown',Regel);
               Continue;
          END;

          IF (Keyword = 'LISTPASSIVE') THEN
          BEGIN
               ListRec.Passive:=RegelYesNo2Boolean ('ListPassive',Regel);
               Continue;
          END;

          IF (Keyword = 'LISTNAMEINSUBJ') THEN
          BEGIN
               ListRec.NameInSubject:=RegelYesNo2Boolean ('ListNameInSubj',Regel);
               Continue;
          END;

          IF (Keyword = 'LISTPRIORITY') THEN
          BEGIN
               ListRec.MLAddress:=Regel2ListPriority (Regel);
               Continue;
          END;

          IF (Keyword = 'LISTDEFAULTACCESS') THEN
          BEGIN
               ListRec.DefaultAccess:=Regel2ListAccess (Regel);
               Continue;
          END;

          IF (Keyword = 'LISTCONFIRMINTERVAL') THEN
          BEGIN
               Val (Regel,ListRec.ConfirmInterval,Nop);
               Continue;
          END;

          IF (Keyword = 'LISTTOAREA') THEN
          BEGIN
               ListRec.ListToArea:=RegelYesNo2Boolean ('ListToArea',Regel);
               Continue;
          END;

          IF (Keyword = 'AREATOLIST') THEN
          BEGIN
               ListRec.AreaToList:=RegelYesNo2Boolean ('AreaToList',Regel);
               Continue;
          END;

          IF (Keyword = 'LISTAREA') THEN
          BEGIN
               ListRec.AreaName:=Regel;
               Continue;
          END;

          IF (Keyword = 'WRITELIST') THEN
          BEGIN
               Flex_MailingList_Write (ListPos,ListRec);
               Continue;
          END;

          IF (Keyword = 'NEWLISTUSER') THEN
          BEGIN
               FillChar (ListUserRec,SizeOf (ListUserRecord),0);

               WITH ListUserRec DO
               BEGIN
                    UserType:=Regel2ListUserType (Regel);
                    ListID:=ListRec.ListID;
                    FidoSplit ('0',Aka);
                    User:='';
                    Email:='';
                    Access:=ListRec.DefaultAccess;
                    SubscribedDate:=0;
                    ConfirmedDate:=0;
                    ConfirmReqDate:=0;
                    ConfirmState:=lcNever;
                    ConfirmCode:=0;
               END; { with }

               ListUserRecNr:=ListUser_WriteNew (ListUserRec);

               Continue;
          END;

          IF (Keyword = 'LISTUSERNAME') THEN
          BEGIN
               ListUserRec.User:=Regel;
               Continue;
          END;

          IF (Keyword = 'LISTUSERAKA') THEN
          BEGIN
               FidoSplit (Regel,ListUserRec.Aka);
               Continue;
          END;

          IF (Keyword = 'LISTUSEREMAIL') THEN
          BEGIN
               ListUserRec.Email:=Regel;
               Continue;
          END;

          IF (Keyword = 'WRITELISTUSER') THEN
          BEGIN
               ListUser_Write (ListUserRecNr,ListUserRec);
               Continue;
          END;

          IF (Keyword = 'NEWAREAFIXFWD') THEN
          BEGIN
               Flex_AreaFixFwd_New (FlexPos,AreaFixFwd);
               Continue;
          END;

          IF (Keyword = 'AREAFIXFWDPATH') THEN
          BEGIN
               AreaFixFwd.AreaListPath:=Regel;
               Continue;
          END;

          IF (Keyword = 'AREAFIXFWDTYPE') THEN
          BEGIN
               AreaFixFwd.AreaList:=Regel2AreaFixFwdListType (Regel);
               Continue;
          END;

          IF (Keyword = 'AREAFIXFWDUNCONDITIONAL') THEN
          BEGIN
               AreaFixFwd.Unconditional:=RegelYesNo2Boolean ('AreaFixFwdUnconditional',Regel);
               Continue;
          END;

          IF (Keyword = 'AREAFIXFWDAKA') THEN
          BEGIN
               FidoSplit (Regel,AreaFixFwd.UplinkAddress);
               Continue;
          END;

          IF (Keyword = 'AREAFIXFWDNAME') THEN
          BEGIN
               AreaFixFwd.AreaManager:=Regel;
               Continue;
          END;

          IF (Keyword = 'AREAFIXFWDPASSWORD') THEN
          BEGIN
               AreaFixFwd.Password:=Regel;
               Continue;
          END;

          IF (Keyword = 'AREAFIXFWDGROUP') THEN
          BEGIN
               AreaFixFwd.Group:=Regel2OneGroupNr (Regel);
               Continue;
          END;

          IF (Keyword = 'AREAFIXFWDADDPLUS') THEN
          BEGIN
               AreaFixFwd.AddPlus:=RegelYesNo2Boolean ('AreaFixFwdAddPlus',Regel);
               Continue;
          END;

          {
          IF (Keyword = 'AREAFIXFWDEXPORTAKA') THEN
          BEGIN
               AreaFixFwd.
               Continue;
          END;
          }

          IF (Keyword = 'WRITEAREAFIXFWD') THEN
          BEGIN
               Flex_AreaFixFwd_Write (FlexPos,AreaFixFwd);
               Continue;
          END;

          IF (Keyword = 'NOEMPTYTEAR') THEN
          BEGIN
               Config.EmptyTear:=FALSE;
               Continue;
          END;

          IF (Keyword = 'NETMAILDECODE') THEN
          BEGIN
               Config.NetmailDecode:=TRUE;
               Continue;
          END;

          IF (Keyword = 'NETMAILDECODEPATH') THEN
          BEGIN
               Config.NetmailDecodePath:=Regel;
               Continue;
          END;

          Log ('SetupCfg: Unknown keyword '+Keyword+' (line '+Word2String (LineNr)+')');
     END; { while }

     Close (InFile);

     FoundConfigPath:=CorrectPath (Config.SystemDir)+'WTRCFG.TDB';
     SaveConfigFile;

     Flex_Done;
     CloseDatabases;
END;


VAR LogPath : STRING;
    IORes   : BYTE;

BEGIN
     WorkDrive:=GetEnv ('WORKDRIVE');
     IF (WorkDrive = '') THEN
     BEGIN
          WriteLn (#7,'WORKDRIVE is not set');
          Halt;
     END;

     LogPath:=GetEnv ('WGTESTLOG');
     IF (LogPath = '') THEN
     BEGIN
          WriteLn (#7,'WGTESTLOG is not set');
          Halt;
     END;

     IF (ParamCount <> 1) THEN
     BEGIN
          WriteLn (#7,'Usage: SetupCfg <path to config.txt>');
          Halt;
     END;

     Assign (LogFile,LogPath);
     {$I-} Append (LogFile); {$I+} IORes:=IOResult;
     IF (IORes <> 0) THEN
     BEGIN
          WriteLn (#7,'Error appending to '+LogPath,' (error ',IORes,')');
          Halt;
     END;

     ReadConfigTxt (FExpand (ParamStr (1)));

     Close (LogFile);
END.
